# adminbot/api_helpers.py

import aiohttp
import json
from typing import Any, List, Dict, Tuple
from urllib.parse import urljoin
from mautrix.types import RoomID
from mautrix.client import Client as MatrixClient
from mautrix.api import Method
from mautrix.errors import MatrixRequestError
from maubot import Plugin
from .utils import normalize_synapse_url

# --- SUPPRIMÉ : La fonction fetch_matrix_room_aliases a été retirée ---
# Cette fonction était moins fiable que la lecture directe de l'état du salon
# et n'est plus utilisée par la commande de migration.

# --- FONCTION EXISTANTE (inchangée) ---
async def fetch_synapse_admin_users(
    client: Plugin, 
    log: Any, 
    http_session: aiohttp.ClientSession, 
    room_id: RoomID, 
    synapse_base_url_raw: str, 
    synapse_admin_token: str, 
    command_format: str
) -> Tuple[List[Dict[str, Any]] | None, str | None]:
    """
    Exécute la requête API Synapse Admin pour récupérer la liste des utilisateurs.
    """
    
    if not synapse_base_url_raw or not synapse_base_url_raw.strip():
        await client.send_text(room_id, f"❌ Erreur de format : L'`URL` du homeserver est manquante. Format attendu : {command_format}")
        return None, None
        
    if not synapse_admin_token or not synapse_admin_token.strip():
        await client.send_text(room_id, f"❌ **Erreur de format :** Le `TOKEN` Administrateur est manquant. Format attendu : {command_format}")
        return None, None

    synapse_base_url, host_header = normalize_synapse_url(synapse_base_url_raw)
    
    api_path = "/_synapse/admin/v2/users" 
    full_url_with_params = urljoin(synapse_base_url, f"{api_path}?from=0&limit=100000")
    
    log.warning(f"⚠️ Requête Admin API vers {synapse_base_url}...")

    try:
        headers = { 'Authorization': f'Bearer {synapse_admin_token}', 'Host': host_header }
        
        async with http_session.get(full_url_with_params, headers=headers) as response:
            response_body = None
            try: response_body = await response.json()
            except (aiohttp.ContentTypeError, json.JSONDecodeError): pass
            
            response.raise_for_status() 

            if response_body and 'users' in response_body:
                return response_body['users'], synapse_base_url
            else:
                await client.send_text(room_id, "❌ Réponse inattendue de l'API Synapse. Le format des données est incorrect ou la liste 'users' est vide.")
                return None, None
                
    except aiohttp.ClientResponseError as e:
         status_code = e.status
         log.error(f"Erreur API aiohttp ({status_code}): {e}. Corps: {response_body}")
         if status_code in (401, 403): await client.send_text(room_id, f"❌ **Erreur d'autorisation ({status_code})** sur **{synapse_base_url}** : Le `TOKEN` est **invalide** ou n'a pas les privilèges administrateur.")
         elif status_code == 404: await client.send_text(room_id, f"❌ **Erreur 404** : Point de terminaison de l'API Synapse Admin non trouvé sur **{synapse_base_url}**.")
         else: await client.send_text(room_id, f"❌ Erreur API ({status_code}) : Impossible de récupérer les données.")
         return None, None
            
    except aiohttp.ClientConnectorError:
        log.error(f"Erreur de connexion : Impossible de se connecter à {synapse_base_url}", exc_info=True)
        await client.send_text(room_id, f"❌ **Erreur de connexion :** Impossible de se connecter à **{synapse_base_url}**.")
        return None, None
    except Exception as e:
        log.error(f"Erreur interne inattendue dans l'helper API: {e}", exc_info=True)
        await client.send_text(room_id, f"❌ **Erreur interne inattendue :** {type(e).__name__}. Voir les logs du bot.")
        return None, None
