# adminbot/commands/helpers/client_manager.py - Gestionnaire de clients temporaires (Corrigé et Fiable)

from typing import Optional, Tuple, TYPE_CHECKING
from mautrix.client import Client as MatrixClient
from mautrix.types import RoomID, MessageType, TextMessageEventContent, Format
from mautrix.errors import MUnknownToken, MatrixRequestError

if TYPE_CHECKING:
    from ...__init__ import AdminBot

async def create_temp_client(
    log: "AdminBot.log",
    user_id: str,
    token: str
) -> Tuple[Optional[MatrixClient], Optional[str]]:
    """
    Crée et valide un client Matrix temporaire pour un utilisateur.
    Chaque client gère sa propre session HTTP.
    Retourne (client, None) en cas de succès, ou (None, message_erreur) en cas d'échec.
    """
    client = None
    try:
        _, homeserver = user_id.split(":", 1)
        
        client = MatrixClient(base_url=f"https://{homeserver}", token=token)
        
        whoami = await client.whoami()

        # NOUVEAU : Assignation critique de l'identifiant utilisateur à l'objet client.
        # C'est cette ligne qui corrige le bug où `client.mxid` était `None`.
        client.mxid = whoami.user_id
        
        if whoami.user_id != user_id:
            await client.api.session.close()
            return None, f"Le token ne correspond pas à l'utilisateur {user_id} (reçu {whoami.user_id})."
        
        log.info(f"Client temporaire créé et vérifié pour {user_id}.")
        return client, None
        
    except (MUnknownToken, MatrixRequestError) as e:
        if client: await client.api.session.close() # S'assure de fermer en cas d'erreur
        if isinstance(e, MUnknownToken) or (isinstance(e, MatrixRequestError) and e.status == 401):
             return None, "Le token fourni est **invalide ou a expiré** (Erreur 401)."
        if isinstance(e, MatrixRequestError):
            return None, f"Erreur de l'API Matrix ({e.status}): {e.message}."
        return None, f"Erreur inattendue : {e}" # Fallback
    except Exception as e:
        if client: await client.api.session.close() # S'assure de fermer en cas d'erreur
        log.error(f"Échec de la création du client temporaire pour {user_id}: {e}", exc_info=True)
        return None, f"Une erreur de connexion ou de configuration est survenue. ({type(e).__name__})"

async def send_formatted_message(plugin: "AdminBot", room_id: RoomID, plain_text: str, html_text: str) -> None:
    """
    Envoie un message formaté (texte + HTML) dans un salon.
    """
    content = TextMessageEventContent(
        msgtype=MessageType.TEXT,
        body=plain_text,
        format=Format.HTML,
        formatted_body=html_text
    )
    await plugin.client.send_message(room_id, content)
