# adminbot/commands/helpers/room_processor.py
# Ce module contient la logique critique pour analyser et traiter un unique salon Matrix.

from typing import Any, List, Dict, Optional, TYPE_CHECKING
import asyncio
from mautrix.client import Client as MatrixClient
from mautrix.types import RoomID, EventType, StateEvent, UserID
from mautrix.errors import MatrixRequestError, MNotFound, MLimitExceeded, MForbidden

# Import de la fonction utilitaire pour les niveaux de pouvoir
from .power_level_utils import manage_power_level_transfer

if TYPE_CHECKING:
    from ...__init__ import AdminBot

async def process_room(
    plugin: "AdminBot", r_id: RoomID, source_client: MatrixClient, dest_client: MatrixClient,
    source_user_id_from_command: str, destination_user_id: UserID,
    dest_joined_rooms: set, dry_run: bool, leave_source_rooms: bool,
    source_hs: str
) -> Dict[str, Any]:
    """
    Analyse et traite un seul salon, puis retourne le dictionnaire de résultat.
    """
    room_name: Optional[str] = None
    is_tombstoned = False
    canonical_alias: Optional[str] = None
    alt_aliases: List[str] = []

    try:
        # Étape 1 : Récupérer les informations de base
        full_state: List[StateEvent] = await source_client.get_state(r_id)
        for state_event in full_state:
            if state_event.type == EventType.ROOM_TOMBSTONE:
                is_tombstoned = True
            elif state_event.type == EventType.ROOM_NAME:
                room_name = getattr(state_event.content, 'name', None)

        try:
            alias_event_content = await source_client.get_state_event(r_id, EventType.ROOM_CANONICAL_ALIAS)
            if alias_event_content:
                if hasattr(alias_event_content, 'canonical_alias'):
                    canonical_alias = alias_event_content.canonical_alias
                    alt_aliases = getattr(alias_event_content, 'alt_aliases', [])
                elif isinstance(alias_event_content, dict):
                    canonical_alias = alias_event_content.get('alias')
                    alt_aliases = alias_event_content.get('alt_aliases', [])
        except MNotFound:
            pass

    except MatrixRequestError as e:
        plugin.log.warning(f"Impossible de récupérer l'état de {r_id}: {e}")

    result: Dict[str, Any] = {
        "room_id": r_id, "name": room_name, "canonical_alias": canonical_alias, "alt_aliases": alt_aliases or []
    }

    # Cas d'ignorance (inchangés)
    if r_id in dest_joined_rooms:
        return result | {"status_key": "IGNORED_ALREADY_JOINED", "details_keys": []}
    if is_tombstoned:
        return result | {"status_key": "IGNORED_OLD_ROOM", "details_keys": []}
    if len(await source_client.get_joined_members(r_id)) <= 2:
        return result | {"status_key": "IGNORED_DM", "details_keys": []}

    # ##################################################################
    # ### SECTION Logique transactionnelle robuste ###
    # ##################################################################

    details = []
    max_retries = 3
    retry_delay = 360.0

    for attempt in range(max_retries):
        try:
            # --- Logique de simulation (DRY-RUN) ---
            if dry_run:
                try:
                    # Tente d'inviter en priorité
                    if "INVITE" not in details:
                        await source_client.invite_user(r_id, destination_user_id, reason="[SIMULATION] Test d'invitation.")
                        details.append("INVITE")
                    await asyncio.sleep(1.5)
                    # MODIFIÉ: Utilisation de la nouvelle clé unifiée "ROLLBACK_SIMULATION"
                    if "ROLLBACK_SIMULATION" not in details:
                        await dest_client.leave_room(r_id)
                        details.append("ROLLBACK_SIMULATION")
                except MForbidden as e:
                    # Logique de fallback si l'invitation est interdite
                    if "You don't have permission to invite users" in str(e):
                        plugin.log.warning(f"SIMULATION: Invitation échouée pour {r_id} (pas de permission). Tentative de jonction/départ direct.")
                        if "JOIN_FALLBACK" not in details:
                            await dest_client.join_room(r_id, servers=[source_hs])
                            details.append("JOIN_FALLBACK")
                        await asyncio.sleep(1.5)
                        # MODIFIÉ: Utilisation de la nouvelle clé unifiée "ROLLBACK_SIMULATION"
                        if "ROLLBACK_SIMULATION" not in details:
                            await dest_client.leave_room(r_id)
                            details.append("ROLLBACK_SIMULATION")
                    else:
                        raise e # Lève à nouveau l'erreur si ce n'est pas celle attendue

                power_sync_action, power_sync_log = await manage_power_level_transfer(
                    plugin, r_id, source_client, dest_client,
                    source_user_id_from_command, destination_user_id,
                    leave_source_rooms, dry_run=True
                )
                plugin.log.info(f"[SIMULATION] RÉSULTAT POUVOIRS pour {r_id}: {power_sync_log}")

                if power_sync_action == "POWER_FAILURE":
                    result.update({"status_key": "FAILURE", "reason": "Erreur de synchro. des pouvoirs", "details_keys": details})
                    return result
                if power_sync_action:
                    details.append(power_sync_action)
                if leave_source_rooms:
                    details.append("LEAVE")
                result.update({"status_key": "SUCCESS_SIMULATED", "details_keys": details})
                return result

            # --- Logique d'exécution réelle ---
            else:
                try:
                    # Tente d'inviter en priorité
                    if "INVITE" not in details:
                        await source_client.invite_user(r_id, destination_user_id, reason="Invitation for user account migration.")
                        details.append("INVITE")
                    await asyncio.sleep(1)
                    if "ACCEPT" not in details:
                        await dest_client.join_room(r_id, servers=[source_hs])
                        details.append("ACCEPT")
                except MForbidden as e:
                     # Logique de fallback si l'invitation est interdite
                    if "You don't have permission to invite users" in str(e):
                        plugin.log.warning(f"EXÉCUTION: Invitation échouée pour {r_id} (pas de permission). Tentative de jonction directe.")
                        if "JOIN_FALLBACK" not in details:
                            await dest_client.join_room(r_id, servers=[source_hs])
                            details.append("JOIN_FALLBACK")
                    else:
                        raise e # Lève à nouveau l'erreur si ce n'est pas celle attendue

                power_sync_action, power_sync_log = await manage_power_level_transfer(
                    plugin, r_id, source_client, dest_client,
                    source_user_id_from_command, destination_user_id,
                    leave_source_rooms, dry_run=False
                )
                plugin.log.info(f"RÉSULTAT POUVOIRS pour {r_id}: {power_sync_log}")

                if power_sync_action == "POWER_FAILURE":
                    result.update({"status_key": "FAILURE", "reason": "Erreur de synchro. des pouvoirs", "details_keys": details})
                    return result
                if power_sync_action:
                    details.append(power_sync_action)
                if leave_source_rooms:
                    await source_client.leave_room(r_id)
                    details.append("LEAVE")
                result.update({"status_key": "SUCCESS", "details_keys": details})
                return result

        except MLimitExceeded as e:
            plugin.log.debug(f"RATE LIMIT DIAGNOSTIC: Contenu de l'exception MLimitExceeded pour {r_id}: {vars(e)}")
            
            if attempt < max_retries - 1:
                sleep_duration = retry_delay
                use_server_delay = hasattr(e, 'retry_after_ms') and e.retry_after_ms is not None and e.retry_after_ms > 0

                if use_server_delay:
                    sleep_duration = (e.retry_after_ms / 1000) + 1.5
                    plugin.log.warning(
                        f"RATE LIMIT: Limite atteinte pour {r_id} (tentative {attempt + 1}/{max_retries}). "
                        f"Le serveur suggère une attente de {e.retry_after_ms}ms. "
                        f"Nouvelle tentative dans {sleep_duration:.2f}s."
                    )
                else:
                    plugin.log.warning(
                        f"RATE LIMIT: Limite atteinte pour {r_id} (tentative {attempt + 1}/{max_retries}). "
                        f"Le serveur n'a pas fourni de délai (pas de 'Retry-After'). "
                        f"Nouvelle tentative dans {sleep_duration:.2f}s (fallback)."
                    )

                await asyncio.sleep(sleep_duration)
                retry_delay *= 2 
                continue
            else:
                plugin.log.error(f"RATE LIMIT: Échec final pour {r_id} après {max_retries} tentatives.")
                result.update({"status_key": "FAILURE", "reason": "Limite de débit (Rate Limit) dépassée", "details_keys": details})
                return result

        except (MatrixRequestError, RuntimeError) as e:
            reason = str(e) if not hasattr(e, 'message') else e.message
            plugin.log.error(f"DIAGNOSTIC: Échec de la transaction pour {r_id}. Raison: {reason}", exc_info=True)
            result.update({"status_key": "FAILURE", "reason": reason, "details_keys": details})
            return result

    return result
