# adminbot/commands/migrate.py - ORCHESTRATEUR DE LA MIGRATION D'UTILISATEUR

import asyncio
from typing import Optional, Tuple, TYPE_CHECKING
from urllib.parse import urlparse
from mautrix.client import Client as MatrixClient
from mautrix.types import RoomID

if TYPE_CHECKING:
    from ..__init__ import AdminBot

from ..utils import format_migration_report
from .helpers.client_manager import create_temp_client, send_formatted_message
from .helpers.room_processor import process_room

# MODIFIÉ : Taille des lots réduite pour des rapports plus fréquents
REPORT_BATCH_SIZE = 5

# --- FONCTION D'INITIALISATION (inchangée) ---
async def _initialize_migration(
    plugin: "AdminBot", room_id: RoomID, source_user_id: str, source_token: str,
    destination_user_id: str, destination_token: str, options_str: str
) -> Tuple[Optional[MatrixClient], Optional[MatrixClient], bool, bool]:
    """Prépare la migration, crée les clients, teste la fédération et envoie le message initial."""
    options_list = options_str.split()
    dry_run = "--do-it" not in options_list
    leave_source_rooms = "--leave" in options_list

    initial_message = (
        f"🔄 **Lancement de la migration...**\n"
        f"- **Source :** `{source_user_id}`\n"
        f"- **Destination :** `{destination_user_id}`\n"
        f"- **Mode :** `{'DRY-RUN (Simulation)' if dry_run else 'EXÉCUTION RÉELLE'}`\n"
        f"- **Quitter les salons source :** `{'Oui' if leave_source_rooms else 'Non'}`"
    )
    await plugin.client.send_text(room_id, initial_message)

    source_client, source_error = await create_temp_client(plugin.log, source_user_id, source_token)
    dest_client, dest_error = await create_temp_client(plugin.log, destination_user_id, destination_token)

    if source_error or dest_error:
        error_report = "❌ **Échec de l'authentification.** Impossible de démarrer la migration.\n"
        if source_error: error_report += f"- **Source (`{source_user_id}`) :** {source_error}\n"
        if dest_error: error_report += f"- **Destination (`{destination_user_id}`) :** {dest_error}\n"
        await plugin.client.send_text(room_id, error_report + "\nVeuillez générer de nouveaux access tokens et réessayer.")
        
        if source_client: await source_client.api.session.close()
        if dest_client: await dest_client.api.session.close()
        
        return None, None, False, False

    source_hs = urlparse(str(source_client.api.base_url)).hostname
    dest_hs = urlparse(str(dest_client.api.base_url)).hostname

    if source_hs != dest_hs:
        plugin.log.info(f"Détection d'une migration fédérée de {source_hs} vers {dest_hs}. Lancement du test de communication...")
        await plugin.client.send_text(room_id, f"ℹ️ Test de la communication (fédération) entre `{source_hs}` et `{dest_hs}`...")
        try:
            plugin.log.info(f"Test de fédération en cours : demande du profil de {destination_user_id} depuis {source_hs} (timeout de 90s)...")
            await asyncio.wait_for(
                source_client.get_profile(destination_user_id),
                timeout=90.0
            )
            
            plugin.log.info(f"✅ Test de fédération réussi vers {dest_hs}. La récupération du profil a fonctionné.")
            await plugin.client.send_text(room_id, f"✅ La communication entre les serveurs est fonctionnelle.")

        except asyncio.TimeoutError:
            plugin.log.error(f"Échec du test de fédération de {source_hs} vers {dest_hs}: Timeout (90s) dépassé.")
            error_report = (
                f"❌ **Échec du Test de Fédération (Timeout)** ❌\n\n"
                f"Le serveur source `{source_hs}` n'a pas réussi à communiquer avec `{dest_hs}` dans le temps imparti (90 secondes). La migration ne peut pas continuer.\n\n"
                "**Que faire ?**\n"
                "Ce problème indique une fédération très lente ou bloquée. Même si vos tests manuels fonctionnent, le bot requiert une communication plus réactive. Vérifiez :\n"
                "1.  La charge et les performances des deux serveurs Synapse.\n"
                "2.  Les logs des deux serveurs pour des erreurs de connexion sortante/entrante."
            )
            await plugin.client.send_text(room_id, error_report)
            if source_client: await source_client.api.session.close()
            if dest_client: await dest_client.api.session.close()
            return None, None, False, False

        except Exception as e:
            plugin.log.error(f"Échec critique du test de fédération de {source_hs} vers {dest_hs}: {e}", exc_info=True)
            error_report = (
                f"❌ **Échec Critique du Test de Fédération** ❌\n\n"
                f"Le serveur source `{source_hs}` n'a pas réussi à communiquer avec le serveur de destination `{dest_hs}`. "
                f"La migration ne peut pas continuer.\n\n"
                f"**Erreur technique rapportée :** `{str(e)}`\n\n"
                "**Que faire ?**\n"
                "Ce problème ne vient pas du bot, mais de la configuration de vos serveurs. Vérifiez les points suivants :\n"
                "1.  Consultez les logs de votre serveur Synapse `{source_hs}` pour des détails sur l'échec.\n"
                "2.  Assurez-vous qu'aucun pare-feu ou problème de DNS ne bloque la communication.\n"
                "3.  Vérifiez que les certificats TLS/SSL des deux serveurs sont valides."
            )
            await plugin.client.send_text(room_id, error_report)
            
            if source_client: await source_client.api.session.close()
            if dest_client: await dest_client.api.session.close()
            return None, None, False, False

    return source_client, dest_client, dry_run, leave_source_rooms

# --- FONCTION PRINCIPALE (MODIFIÉE) ---
async def execute_migrate(plugin: "AdminBot", room_id: RoomID, source_user_id: str, source_token: str, destination_user_id: str, destination_token: str, options_str: str) -> None:
    clients = await _initialize_migration(plugin, room_id, source_user_id, source_token, destination_user_id, destination_token, options_str)
    source_client, dest_client, dry_run, leave_source_rooms = clients
    if not source_client or not dest_client:
        return

    all_results = []
    try:
        source_hs = urlparse(str(source_client.api.base_url)).hostname
        
        source_joined_rooms = await source_client.get_joined_rooms()
        dest_joined_rooms_set = set(await dest_client.get_joined_rooms())
        total_rooms = len(source_joined_rooms)
        
        if total_rooms == 0:
            await plugin.client.send_text(room_id, "✅ **Migration terminée** : Aucun salon à migrer pour l'utilisateur source.")
            return

        batch_results = []
        for i, r_id in enumerate(source_joined_rooms):
            room_result = await process_room(
                plugin, r_id, source_client, dest_client, 
                source_user_id, destination_user_id, 
                dest_joined_rooms_set, dry_run, leave_source_rooms, source_hs
            )
            all_results.append(room_result)
            batch_results.append(room_result)
            
            if (i + 1) % REPORT_BATCH_SIZE == 0 or (i + 1) == total_rooms:
                if batch_results:
                    batch_start_index = i - len(batch_results) + 1
                    plain, html = format_migration_report(batch_results, total_rooms, batch_start_index, dry_run, leave_source_rooms)
                    await send_formatted_message(plugin, room_id, plain, html)
                    batch_results.clear()
            
            # MODIFIÉ : Délai de base augmenté pour réduire l'agressivité globale
            await asyncio.sleep(7.0)
            
    finally:
        success_count = sum(1 for r in all_results if 'SUCCESS' in r.get("status_key", ""))
        fail_count = len(all_results) - success_count
        summary_title = "Rapport Final de Simulation" if dry_run else "Rapport Final de Migration"
        summary_message = (
            f"🏁 **{summary_title} Terminé**\n"
            f"- ✅ **{success_count}** salon(s) traité(s) avec succès.\n"
            f"- 🚫 **{fail_count}** salon(s) en échec ou ignoré(s)."
        )
        if all_results:
            await plugin.client.send_text(room_id, summary_message)
        
        plugin.log.info("Nettoyage des sessions de connexion des clients temporaires...")
        if source_client:
            try: await source_client.api.session.close()
            except Exception as e: plugin.log.warning(f"Échec de la fermeture de session du client source : {e}")
        if dest_client:
            try: await dest_client.api.session.close()
            except Exception as e: plugin.log.warning(f"Échec de la fermeture de session du client destination : {e}")

        plugin.log.info("Processus de migration terminé.")
