# adminbot/membership.py - LOGIQUE DE JOIN ET CONTRÔLE D'ACCÈS PROACTIF (Corrigé)

from typing import TYPE_CHECKING
from mautrix.types import Membership, StateEvent
from mautrix.errors import MatrixRequestError

if TYPE_CHECKING:
    from .__init__ import AdminBot

# NOUVEAU : Import des fonctions de contrôle d'accès nécessaires
from .access_control import is_authorized, refuse_and_leave

async def handle_room_membership_logic(plugin: 'AdminBot', evt: StateEvent) -> None:
    """
    Gère l'acceptation d'une invitation et effectue le contrôle de sécurité IMMÉDIATEMENT après avoir rejoint.
    """
    # 1. Traiter uniquement les invitations qui concernent le bot lui-même
    if evt.state_key != plugin.bot_id or evt.content.membership != Membership.INVITE:
        return

    room_id = evt.room_id
    
    # 2. Mécanisme de verrouillage pour éviter les traitements en double (inchangé et crucial)
    if room_id in plugin.processing_invites:
        plugin.log.debug(f"ℹ️ DÉDUPLICATION: Traitement d'invitation déjà en cours pour {room_id}. Ignoré.")
        return
    
    # On verrouille la salle. Le verrou sera libéré à la fin de cette fonction.
    plugin.processing_invites.add(room_id)
    plugin.log.info(f"💌 INVITATION REÇUE: De {evt.sender} dans {room_id}. Verrouillage et tentative de join.")
    
    try:
        # POINT CRUCIAL : Le code ATTEND ici la réponse du serveur distant.
        await plugin.client.join_room(room_id)
        plugin.log.info(f"✅ JOIN: Réussi pour {room_id}. Déclenchement du contrôle de sécurité immédiat.")
    except MatrixRequestError as e:
        if "already in room" in e.message.lower() or e.status == 409:
            plugin.log.debug(f"ℹ️ JOIN: Déjà dans la salle {room_id}.")
        else:
            plugin.log.error(f"❌ ERREUR JOIN: Échec pour rejoindre {room_id}: {e}.")
            # CRUCIAL : Si le join échoue, on doit libérer le verrou ici.
            if room_id in plugin.processing_invites:
                plugin.processing_invites.remove(room_id)
            return

    # --- CONTRÔLE DE SÉCURITÉ PROACTIF (Post-Join) ---
    # Ce code s'exécute SEULEMENT APRÈS le succès de `join_room` ci-dessus.
    
    authorized_users = plugin.config.authorized_users_set
    inviter_id = evt.sender

    if not is_authorized(inviter_id, authorized_users):
        # L'inviteur N'EST PAS autorisé. On lance la procédure de refus et de départ.
        plugin.log.warning(f"🚫 ACCÈS REFUSÉ (Post-Join): L'inviteur {inviter_id} n'est pas autorisé. Lancement du départ de {room_id}.")
        await refuse_and_leave(plugin, room_id, inviter_id)
        return
    
    # Si on arrive ici, l'inviteur EST autorisé.
    plugin.log.info(f"✅ ACCÈS AUTORISÉ (Post-Join): L'inviteur {inviter_id} est autorisé dans {room_id}. Envoi du message de bienvenue.")
    
    # Envoi du message de bienvenue
    welcome_message = plugin.config["welcome_message"].format(inviter_id=inviter_id)
    await plugin.client.send_text(room_id, welcome_message)
    
    plugin.welcomed_rooms.add(room_id)
    
    # On libère le verrou.
    if room_id in plugin.processing_invites:
        plugin.processing_invites.remove(room_id)
        plugin.log.debug(f"Déverrouillage du traitement d'invitation pour {room_id}.")
