# adminbot/room_checks.py - LOGIQUE PROACTIVE DE VÉRIFICATION DES SALLES (Version finale et compatible)

from mautrix.types import EventType, RoomID, UserID, Membership, StateEvent
from mautrix.errors import MatrixRequestError
import asyncio
from typing import TYPE_CHECKING, List

if TYPE_CHECKING:
    from .__init__ import AdminBot

async def proactive_check_rooms(plugin: 'AdminBot') -> None:
    """
    Vérifie toutes les salles jointes : quitte celles où le bot est seul,
    puis vérifie les permissions des restantes.
    """
    try:
        joined_rooms = await plugin.client.get_joined_rooms()
        rooms_to_check: List[RoomID] = []
        leave_tasks = []

        plugin.log.info(f"Début de l'analyse proactive des {len(joined_rooms)} salles jointes.")

        for room_id in joined_rooms:
            try:
                # --- MODIFICATION DE LA LOGIQUE DE COMPTAGE ---
                # MODIFIÉ : Remplacement de l'appel direct par une méthode plus fondamentale et compatible.
                # 1. On récupère tous les événements d'état de la salle.
                all_state_events = await plugin.client.get_state(room_id)
                
                # 2. On filtre manuellement pour ne compter que les membres réellement joints.
                joined_members_count = 0
                for event in all_state_events:
                    if isinstance(event, StateEvent) and event.type == EventType.ROOM_MEMBER:
                        if event.content.membership == Membership.JOIN:
                            joined_members_count += 1
                
                member_count = joined_members_count
                # --- FIN DE LA MODIFICATION ---

                if member_count <= 1:
                    plugin.log.info(f"🧹 Salle orpheline détectée {room_id} (membres: {member_count}). Préparation au départ.")
                    leave_tasks.append(_proactive_leave_room(plugin, room_id))
                else:
                    plugin.log.info(f"ℹ️ Salle valide détectée {room_id} (membres: {member_count}). Planification de la vérification des permissions.")
                    rooms_to_check.append(room_id)

            except MatrixRequestError as e:
                plugin.log.warning(f"⚠️ Erreur API lors du traitement de la salle {room_id}: {e.__class__.__name__} ({e.status}). Saut de la vérification.")
            except Exception as e:
                plugin.log.error(f"Erreur inattendue lors du traitement de la salle {room_id}: {e.__class__.__name__}: {e}", exc_info=True)

        if leave_tasks:
            semaphore_limit = plugin.proactive_limiter._value
            plugin.log.info(f"Lancement du départ de {len(leave_tasks)} salles orphelines avec une limite de {semaphore_limit} tâches concurrentes et un délai de 5s entre chaque.")
            await asyncio.gather(*leave_tasks, return_exceptions=True)
            plugin.log.info("Fin du processus de nettoyage des salles orphelines.")

        if rooms_to_check:
            plugin.log.info(f"Vérification des niveaux de puissance pour {len(rooms_to_check)} salles restantes.")
            await asyncio.gather(*[
                _check_room_permissions(plugin, room_id) for room_id in rooms_to_check
            ])
            plugin.log.info("Vérification proactive des salles terminée.")
        else:
            plugin.log.info("Aucune salle restante à vérifier pour les niveaux de puissance.")

    except Exception as e:
        plugin.log.error(f"Échec critique de la vérification proactive des salles: {e}")


async def _proactive_leave_room(plugin: 'AdminBot', room_id: RoomID):
    """Quitte la salle en respectant la limite de débit."""
    async with plugin.proactive_limiter:
        try:
            await asyncio.sleep(5)
            await plugin.client.leave_room(room_id)
            plugin.log.info(f"✅ Départ réussi de la salle orpheline {room_id}.")
        except MatrixRequestError as e:
            plugin.log.warning(f"⚠️ Échec du départ de {room_id} (Code {e.status}).")
        except Exception as e:
            plugin.log.error(f"Erreur inattendue lors du départ de {room_id}: {e}")


async def _check_room_permissions(plugin: 'AdminBot', room_id: RoomID):
    """Vérifie que le bot a les droits d'admin/modération nécessaires dans la salle donnée."""
    try:
        content = await plugin.client.get_state_event(room_id, EventType.ROOM_POWER_LEVELS, "")
        users_power = content.users.get(plugin.bot_id, content.users_default)
        required_power = max(content.kick, content.ban, content.redact)

        if required_power < 50:
            required_power = 50

        if users_power < required_power:
            plugin.log.warning(f"⚠️ Bot a un niveau de puissance insuffisant ({users_power}) dans la salle {room_id}. Minimum recommandé: {required_power}.")
        else:
            plugin.log.debug(f"✅ Bot a un niveau de puissance suffisant ({users_power}) dans la salle {room_id}.")

    except MatrixRequestError as e:
        plugin.log.error(f"Erreur lors de la vérification de la salle {room_id}: {e.__class__.__name__} ({e.status}) : {e.message}")
    except Exception as e:
        plugin.log.error(f"Erreur inattendue lors de la vérification de la salle {room_id}: {e.__class__.__name__}: {e}")
