# adminbot/__init__.py - LOGIQUE DE PLUGIN PRINCIPALE (Déclenchement au démarrage)

import aiohttp
import asyncio
from typing import Type, Set
from maubot import Plugin, MessageEvent
from maubot.handlers import event
from mautrix.types import EventType, StateEvent, UserID, RoomID

from .config import AdminConfig
from .event_handlers import handle_room_message
from .membership import handle_room_membership_logic
from .room_checks import proactive_check_rooms


class AdminBot(Plugin):
    config: AdminConfig
    bot_id: UserID
    http_session: aiohttp.ClientSession
    proactive_limiter: asyncio.Semaphore
    processing_invites: Set[RoomID]
    welcomed_rooms: Set[RoomID]
    # SUPPRIMÉ : Le drapeau pour la vérification initiale n'est plus nécessaire.
    # initial_check_done: bool

    @classmethod
    def get_config_class(cls) -> Type[AdminConfig]:
        return AdminConfig

    async def start(self) -> None:
        await super().start()
        
        self.on_external_config_update()
        
        self.bot_id = self.client.mxid
        self.http_session = aiohttp.ClientSession()
        self.proactive_limiter = asyncio.Semaphore(5)
        self.processing_invites = set()
        self.welcomed_rooms = set()
        # SUPPRIMÉ : L'initialisation du drapeau est retirée.
        # self.initial_check_done = False
        
        self.log.info("✅ READY: AdminBot est en ligne et prêt à recevoir des événements.")

        # NOUVEAU : Lancement direct et fiable de la tâche de vérification proactive au démarrage.
        # La tâche s'exécutera en arrière-plan sans bloquer le bot.
        self.loop.create_task(proactive_check_rooms(self))

    def on_external_config_update(self) -> None:
        self.log.info("🔄 CONFIGURATION : Chargement et traitement des données...")
        self.config.load_and_update()
        self.config.parse_config()
        self.log.info(f"✅ CONFIGURATION MISE À JOUR : {len(self.config.authorized_users_set)} administrateur(s) sont maintenant actifs.")

    async def stop(self) -> None:
        await super().stop()
        self.log.info("🛑 SHUTDOWN: Fermeture de la session HTTP du plugin AdminBot.")
        if self.http_session and not self.http_session.closed:
            await self.http_session.close()

    # SUPPRIMÉ : La méthode de déclenchement via événement n'est plus utile.
    # async def _trigger_initial_check(self) -> None:
    #     if not self.initial_check_done:
    #         self.initial_check_done = True
    #         self.log.info("✅ ÉVÉNEMENT REÇU : Premier événement détecté. Le bot est synchronisé. Lancement de la vérification proactive des salles.")
    #         self.loop.create_task(proactive_check_rooms(self))

    @event.on(EventType.ROOM_MEMBER)
    async def handle_membership_change(self, evt: StateEvent) -> None:
        # SUPPRIMÉ : L'appel au déclencheur est retiré.
        # await self._trigger_initial_check()
        await handle_room_membership_logic(self, evt)

    @event.on(EventType.ROOM_MESSAGE)
    async def handle_command_dispatch(self, evt: MessageEvent) -> None:
        # SUPPRIMÉ : L'appel au déclencheur est retiré.
        # await self._trigger_initial_check()
        await handle_room_message(self, evt)
