# adminbot/commands/helpers/power_level_utils.py - VERSION CORRIGÉE ET FIABLE

from typing import Tuple, Optional, TYPE_CHECKING
from mautrix.client import Client as MatrixClient
# CORRECTION : Importation de PowerLevelStateEventContent pour pouvoir créer un nouvel objet.
from mautrix.types import RoomID, EventType, UserID, StateEvent, PowerLevelStateEventContent

if TYPE_CHECKING:
    from ...__init__ import AdminBot

async def manage_power_level_transfer(
    plugin: "AdminBot",
    r_id: RoomID,
    source_client: MatrixClient,
    dest_client: MatrixClient,
    source_user_id_from_command: str,
    destination_user_id: UserID,
    leave_source_rooms: bool,
    dry_run: bool
) -> Tuple[Optional[str], str]:
    """
    Analyse les niveaux de pouvoir et décide s'il faut promouvoir l'utilisateur de destination.
    En mode `dry_run`, simule l'action sans l'exécuter.
    
    Retourne un tuple:
    - Une clé d'action ("POWER_SYNC", "POWER_FAILURE" ou None) pour le rapport.
    - Un message de log détaillé décrivant l'opération.
    """
    try:
        # Étape 1: Récupérer les événements d'état des pouvoirs et de la création
        power_levels_event_content = await source_client.get_state_event(r_id, EventType.ROOM_POWER_LEVELS)
        
        full_state: list[StateEvent] = await source_client.get_state(r_id)
        create_event_list = [event for event in full_state if event.type == EventType.ROOM_CREATE]
        
        if not create_event_list:
            raise RuntimeError("L'événement de création (m.room.create) est introuvable pour ce salon.")
            
        create_event = create_event_list[0]
        creator_id = create_event.sender

        # Étape 2: Déterminer si l'utilisateur source est le créateur du salon
        is_creator = (creator_id == source_user_id_from_command)
        
        # Étape 3: Calculer le niveau de pouvoir cible pour l'utilisateur de destination
        users_default_power = power_levels_event_content.users_default
        source_power = power_levels_event_content.users.get(source_user_id_from_command, users_default_power)
        
        target_power_level = source_power
        
        # Cas critique de sécurité
        if is_creator and leave_source_rooms:
            plugin.log.warning(f"🛡️ SÉCURITÉ (OWNER): Le créateur {creator_id} quitte {r_id}. "
                             f"La destination {destination_user_id} sera promue au niveau 100.")
            target_power_level = 100
        
        # Étape 4: Vérifier si une mise à jour est nécessaire
        current_dest_power = power_levels_event_content.users.get(destination_user_id, users_default_power)

        if current_dest_power >= target_power_level:
            log_message = (f"SYNCHRO POUVOIRS: Aucune action requise pour {r_id}. "
                           f"Niveau actuel ({current_dest_power}) >= Niveau visé ({target_power_level}).")
            return None, log_message

        # Étape 5: Exécuter ou simuler la mise à jour
        log_prefix = "[SIMULATION] " if dry_run else ""
        log_message = (f"{log_prefix}SYNCHRO POUVOIRS: Promotion requise pour {destination_user_id} dans {r_id}. "
                       f"Niveau actuel: {current_dest_power}, Niveau visé: {target_power_level}.")
        
        if not dry_run:
            # --- BLOC DE CORRECTION ---
            # Au lieu de power_levels_event_content.copy(), nous reconstruisons un objet propre.
            
            # 1. Copier le dictionnaire des utilisateurs, qui, lui, a bien une méthode .copy()
            new_users_dict = power_levels_event_content.users.copy()
            # 2. Mettre à jour le niveau de pouvoir dans cette copie
            new_users_dict[destination_user_id] = target_power_level
            
            # 3. Créer une nouvelle instance de PowerLevelStateEventContent avec les données à jour.
            new_power_content = PowerLevelStateEventContent(
                ban=power_levels_event_content.ban,
                kick=power_levels_event_content.kick,
                redact=power_levels_event_content.redact,
                invite=power_levels_event_content.invite,
                users_default=power_levels_event_content.users_default,
                events_default=power_levels_event_content.events_default,
                state_default=power_levels_event_content.state_default,
                users=new_users_dict,  # On utilise notre dictionnaire modifié
                events=power_levels_event_content.events,
                notifications=power_levels_event_content.notifications
            )
            # --- FIN DU BLOC DE CORRECTION ---

            await source_client.send_state_event(r_id, EventType.ROOM_POWER_LEVELS, new_power_content)
        
        return "POWER_SYNC", log_message

    except Exception as e:
        log_message = f"ERREUR POUVOIRS: Impossible de traiter les niveaux de pouvoir pour {r_id}: {e}"
        plugin.log.error(log_message, exc_info=True)
        return "POWER_FAILURE", log_message
