# adminbot/commands/lastseens.py

import aiohttp
import time
from typing import Any, List, Tuple
from mautrix.types import RoomID
from maubot import Plugin

from ..api_helpers import fetch_synapse_admin_users
from ..utils import ms_to_days, format_users_to_html_table

async def execute_lastseens(client: Plugin, log: Any, http_session: aiohttp.ClientSession, room_id: RoomID, synapse_base_url_raw: str, days_str: str, synapse_admin_token: str) -> None:
    # ... (partie initiale de la fonction inchangée)
    NOW_MS = int(time.time() * 1000)
    command_format = "`!admin lastseens <SERVER_URL> <JOURS> <TOKEN>`"
    
    try:
        days = int(days_str)
        if days <= 0: raise ValueError("jours <= 0")
        time_limit_ms = NOW_MS - (days * 24 * 60 * 60 * 1000) 
    except ValueError:
        await client.send_text(room_id, f"❌ **Erreur de paramètre :** Le nombre de jours (`JOURS`) doit être un entier positif. Format attendu : {command_format}")
        return

    await client.send_text(room_id, f"⏳ Récupération des données d'activité (dernière vue) pour les **{days}** derniers jours.")

    users_data, synapse_base_url = await fetch_synapse_admin_users(
        client, log, http_session, room_id, synapse_base_url_raw, synapse_admin_token, command_format
    )
    if users_data is None:
        return

    # MODIFIÉ : Le tuple stocke maintenant la durée, l'alias, l'ID et l'activité
    sortable_users: List[Tuple[int, str | None, str, str]] = [] 
    
    for user in users_data:
        last_seen_ts = user.get('last_seen_ts')
        
        if last_seen_ts is not None and last_seen_ts >= time_limit_ms:
            inactivity_duration_ms = NOW_MS - last_seen_ts
            age_str = ms_to_days(inactivity_duration_ms)
            
            display_name = user.get('displayname')
            user_id = user['name']
            
            sortable_users.append((inactivity_duration_ms, display_name, user_id, age_str))

    sortable_users.sort(key=lambda x: x[0], reverse=False) 
    # MODIFIÉ : On passe le tuple (display_name, user_id, activity) au formateur
    active_users_data = [(name, mxid, activity) for _, name, mxid, activity in sortable_users]

    plain_text_response, formatted_html_response = format_users_to_html_table(active_users_data, synapse_base_url, days, type="seen")
    
    await client.send_message(room_id=room_id, content={
        "msgtype": "m.text", "body": plain_text_response, "format": "org.matrix.custom.html", "formatted_body": formatted_html_response
    })
