# adminbot/config.py - Gère la définition et le TRAITEMENT de la configuration (Version Robuste)

from mautrix.util.config import BaseProxyConfig, ConfigUpdateHelper
from typing import Any, Dict, Set
from mautrix.types import UserID

class AdminConfig(BaseProxyConfig):
    """
    Cette classe définit la structure de la configuration du plugin.
    Elle a deux rôles principaux :
    1. Déclarer les champs de configuration à Maubot via `do_update`.
    2. Fournir une méthode pour traiter/transformer les données brutes en un format
       optimisé pour le reste du plugin via `parse_config`.
    """
    # Attribut pour stocker la liste TRAITÉE des utilisateurs autorisés.
    # C'est notre "source de vérité" unique et performante (un Set).
    authorized_users_set: Set[UserID]

    def do_update(self, helper: ConfigUpdateHelper) -> None:
        """
        Définit les champs du formulaire pour l'interface de Maubot.
        Cette méthode est appelée par `load_and_update()` pour savoir
        quelles clés copier depuis le fichier YAML.
        """
        helper.copy("authorized_users")
        helper.copy("welcome_message")
        helper.copy("refusal_message")

    def parse_config(self) -> None:
        """
        Traite les données brutes (préalablement chargées par `load_and_update`)
        pour les rendre directement utilisables par le bot.
        """
        # On transforme la liste de MXIDs en un 'set' de chaînes en minuscules.
        # Un 'set' offre une recherche quasi instantanée (O(1)), ce qui est idéal
        # pour les vérifications d'autorisation fréquentes.
        # La conversion en minuscules garantit une comparaison insensible à la casse.
        raw_users = self.get("authorized_users", [])
        self.authorized_users_set = {user.lower() for user in raw_users}

    @classmethod
    def get_defaults(cls) -> Dict[str, Any]:
        """
        Définit les valeurs par défaut si elles ne sont pas présentes
        dans le fichier de configuration de l'instance.
        """
        return {
            "authorized_users": [],
            "welcome_message": "Session chiffrée établie. Je suis en ligne et prêt. 🔒",
            "refusal_message": "Désolé, l'utilisateur {inviter_id} n'est pas autorisé à m'inviter ici. Je dois quitter.",
        }
