# adminbot/event_handlers.py - DISPATCHER DE COMMANDES SÉCURISÉ (Corrigé)

import re
from typing import TYPE_CHECKING
from mautrix.types import MessageEvent, MessageType

if TYPE_CHECKING:
    from .__init__ import AdminBot

from .access_control import is_authorized
from .commands.lastseens import execute_lastseens
from .commands.lastregisters import execute_lastregisters
from .commands.migrate import execute_migrate
from .commands.help import execute_help

# Expressions régulières
regex_lastseens = re.compile(r"^lastseens\s+([^ ]+)\s+(\d+)\s+(.+)$")
regex_lastregisters = re.compile(r"^lastregisters\s+([^ ]+)\s+(\d+)\s+(.+)$")
regex_migrate = re.compile(r"^migrate from\s+([^ ]+)\s+([^ ]+)\s+to\s+([^ ]+)\s+([^ ]+)\s*(.*)$")


async def handle_room_message(plugin: 'AdminBot', evt: MessageEvent) -> None:
    """
    Gère les messages entrants et effectue le contrôle de sécurité avant de dispatcher.
    """
    if evt.content.msgtype != MessageType.TEXT or not evt.content.body or evt.sender == plugin.bot_id:
        return

    if not is_authorized(evt.sender, plugin.config.authorized_users_set):
        plugin.log.warning(f"🚫 SÉCURITÉ: Commande non autorisée tentée par {evt.sender} dans {evt.room_id}.")
        return

    message_body = evt.content.body.strip()
    if message_body.lower().startswith("!admin"):
        await handle_admin_command(plugin, evt)
    elif message_body.lower() in ("!help", "help"):
        await handle_help_command(plugin, evt)

async def handle_admin_command(plugin: 'AdminBot', evt: MessageEvent) -> None:
    message_body = evt.content.body.strip()
    room_id = evt.room_id
    command_args = message_body[7:].strip()
    plugin.log.info(f"✅ EXÉCUTION: !admin {command_args[:30]}... par {evt.sender}.")
    try:
        if match := regex_lastseens.match(command_args):
            url, days, token = match.groups()
            await execute_lastseens(plugin.client, plugin.log, plugin.http_session, room_id, url, days, token)
        elif match := regex_lastregisters.match(command_args):
            url, days, token = match.groups()
            await execute_lastregisters(plugin.client, plugin.log, plugin.http_session, room_id, url, days, token)
        elif match := regex_migrate.match(command_args):
            source_user, source_token, dest_user, dest_token, options_str = match.groups()
            options_str = options_str.strip() if options_str else ""
            
            # MODIFIÉ : Le log passe en .info pour être toujours visible.
            # C'est un outil de diagnostic crucial pour vérifier la valeur exacte des options capturées.
            plugin.log.info(f"DIAGNOSTIC MIGRATE: Options capturées par la regex : [{options_str}]")
            
            await execute_migrate(plugin, room_id, source_user, source_token, dest_user, dest_token, options_str)
        else:
            await plugin.client.send_text(room_id, f"❌ **Erreur de Commande :** La commande `!admin {command_args[:30]}` n'est pas reconnue ou mal formatée. Utilisez `!help`.")
        
        try:
            await plugin.client.redact(room_id, evt.event_id, reason="Rédaction de commande sensible")
        except Exception as e:
            plugin.log.warning(f"Impossible de rédiger la commande dans {room_id}: {e}")

    except Exception as e:
        plugin.log.error(f"❌ ERREUR CRITIQUE lors de l'exécution de la commande admin: {e}", exc_info=True)
        await plugin.client.send_text(room_id, f"❌ **Erreur Critique :** Une erreur interne est survenue. Voir les logs du bot.")

async def handle_help_command(plugin: 'AdminBot', evt: MessageEvent) -> None:
    plugin.log.info(f"💡 EXÉCUTION: HELP par {evt.sender}.")
    await execute_help(plugin.client, evt.room_id)
    try:
        await plugin.client.redact(evt.room_id, evt.event_id, reason="Rédaction de commande d'aide")
    except Exception:
        pass
