# adminbot/utils.py (Version 15.9 - Légende unifiée pour le rollback)

from typing import List, Tuple, Dict, Any
from urllib.parse import urlparse
import html

# --- MODIFIÉ : Fusion de "Refus" et "Quitter (Test)" en un seul "Rollback".
MIGRATE_LEGEND_ICONS = {
    "📨": "Invitation",
    "🚪": "Jonction (après invite)",
    "➡️🚪": "Jonction Directe (Fallback)",
    "↩️": "Rollback (Test)",
    "⚡️": "Pouvoirs",
    "👋": "Départ",
    "✅": "Succès",
    "⚪": "Ignoré",
    "❌": "Échec"
}

# --- Fonctions de conversion et moteur de rendu générique (inchangés) ---

def normalize_synapse_url(url_raw: str) -> Tuple[str, str]:
    url = url_raw.strip().rstrip('/')
    if not url.startswith(('http://', 'https://')):
        url = 'https://' + url
    parsed_url = urlparse(url)
    host = parsed_url.netloc.split(':')[0]
    return url, host

def ms_to_days(ms: int) -> str:
    seconds = ms // 1000
    days = seconds // (24 * 3600)
    seconds %= (24 * 3600)
    hours = seconds // 3600
    seconds %= 3600
    minutes = seconds // 60
    if days > 0: return f"{days} jours et {hours} heures"
    elif hours > 0: return f"{hours} heures et {minutes} minutes"
    elif minutes > 0: return f"{minutes} minutes"
    else: return "Moins d'une minute"

def format_generic_html_table(
    title_text: str, column_headers: List[str], data_rows: List[List[str]], empty_message: str,
    pre_header_html: str = "", html_safe_columns: List[int] = None
) -> Tuple[str, str]:
    if not data_rows: return empty_message, f"<p>{html.escape(empty_message)}</p>"
    plain_text_header = f"**{title_text}**"
    plain_text_rows = []
    for row in data_rows:
        row_text = []
        for i, cell in enumerate(row):
            clean_cell = html.unescape(cell.replace('<code>', '').replace('</code>', '').replace('<br>', ', ').replace('<span>', '').replace('</span>', '').replace('<b>', '').replace('</b>', ''))
            row_text.append(f"{column_headers[i]}: {clean_cell}")
        plain_text_rows.append("- " + " | ".join(row_text))
    plain_text = f"{plain_text_header}\n" + "\n".join(plain_text_rows)
    html_content = f"<h3>{html.escape(title_text)}</h3>"
    if pre_header_html: html_content += pre_header_html
    html_content += "<table style='width:100%; border-collapse: collapse; border: 1px solid #ddd;'><thead><tr>"
    for header in column_headers:
        html_content += f"<th style='border: 1px solid #ddd; padding: 8px; text-align: left;'>{html.escape(header)}</th>"
    html_content += "</tr></thead><tbody>"
    for row in data_rows:
        html_content += "<tr>"
        for i, cell in enumerate(row):
            cell_content = cell if (html_safe_columns and i in html_safe_columns) else html.escape(str(cell))
            if "ID" in column_headers[i] and not (html_safe_columns and i in html_safe_columns):
                 cell_content = f"<code>{cell_content}</code>"
            html_content += f"<td style='border: 1px solid #ddd; padding: 8px;'>{cell_content}</td>"
        html_content += "</tr>"
    html_content += "</tbody></table>"
    return plain_text, html_content

def format_users_to_html_table(user_data: List[Tuple[str | None, str, str]], base_url: str, days: int, type: str = "seen") -> Tuple[str, str]:
    if type == "created":
        title = f"{len(user_data)} Nouveaux Comptes sur {base_url} (derniers {days} jours)"
        headers = ["Alias", "UserID", "Âge du Compte"]
        action_text = "enregistrés"
    else:
        title = f"{len(user_data)} Utilisateurs Actifs sur {base_url} (derniers {days} jours)"
        headers = ["Alias", "UserID", "Dernière Activité"]
        action_text = "actifs"
    empty_msg = f"Aucun utilisateur {action_text} trouvé sur {base_url} au cours des {days} derniers jours."
    data_rows = [[display_name or '(aucun)', mxid, activity] for display_name, mxid, activity in user_data]
    return format_generic_html_table(title, headers, data_rows, empty_msg)

# --- Formatteur de migration amélioré ---

def format_migration_report(results: List[Dict[str, Any]], total_rooms: int, batch_start_index: int, dry_run: bool, leave_source_rooms: bool) -> Tuple[str, str]:
    processed_count = batch_start_index + len(results)
    title = f"Rapport de Progression - Salons {batch_start_index + 1} à {processed_count} / {total_rooms}"
    headers = ["Salon", "RoomID & Adresses", "Statut", "Détails"]

    header_tags = []
    if dry_run:
        header_tags.append("<span style='background-color: #ffc107; color: #000; padding: 2px 6px; border-radius: 4px; font-weight: bold;'>SIMULATION</span>")
    if not leave_source_rooms:
        header_tags.append("<span style='background-color: #17a2b8; color: #fff; padding: 2px 6px; border-radius: 4px; font-weight: bold;'>MAINTIEN SOURCE</span>")
    
    header_info_html = ""
    if header_tags:
        header_info_html = f"<div style='margin-bottom: 10px;'><b>Mode :</b> {' '.join(header_tags)}</div>"

    legend_items = " | ".join([f"{icon} {desc}" for icon, desc in MIGRATE_LEGEND_ICONS.items()])
    legend_html = (
        f"<div style='font-size: 0.9em; margin-bottom: 10px; padding: 5px; border: 1px solid #ccc; border-radius: 4px;'>"
        f"<b>Légende :</b> {legend_items}"
        "</div>"
    )

    data_rows = []
    for r in results:
        status_map = {
            "SUCCESS": "✅", "SUCCESS_SIMULATED": "✅", "IGNORED_DM": "⚪ (DM)", "IGNORED_OLD_ROOM": "⚪ (old room)",
            "IGNORED_ALREADY_JOINED": "⚪ (déjà membre)", "FAILURE": "❌",
        }
        status_key = r.get("status_key", "FAILURE")
        status_html = status_map.get(status_key, "❌")
        if status_key == "FAILURE" and r.get("reason"): status_html += f" <span style='color:red;'>({html.escape(r.get('reason'))})</span>"

        # --- MODIFIÉ : Utilisation de la nouvelle clé unique "ROLLBACK_SIMULATION".
        details_map = { 
            "INVITE": "📨", 
            "ACCEPT": "🚪", 
            "POWER_SYNC": "⚡️", 
            "LEAVE": "👋", 
            "JOIN_FALLBACK": "➡️🚪",
            "ROLLBACK_SIMULATION": "↩️"
        }
        details_icons = ' '.join(details_map.get(key, '') for key in r.get("details_keys", []))

        combined_id_cell = f"<code>{html.escape(r['room_id'])}</code>"
        canonical_alias = r.get('canonical_alias')
        alt_aliases = r.get('alt_aliases', [])
        
        if canonical_alias:
            combined_id_cell += f"<br><b><code>{html.escape(canonical_alias)}</code></b>"
        
        if alt_aliases:
            other_aliases = [alias for alias in alt_aliases if alias != canonical_alias]
            if other_aliases:
                aliases_html = "<br>".join([f"<code>{html.escape(a)}</code>" for a in other_aliases])
                combined_id_cell += f"<br>{aliases_html}"

        data_rows.append([ r.get('name') or '(aucun)', combined_id_cell, status_html, details_icons ])
    
    combined_header_html = header_info_html + legend_html
    
    return format_generic_html_table(
        title, headers, data_rows, "Aucun salon à rapporter dans ce lot.",
        pre_header_html=combined_header_html, html_safe_columns=[1, 2, 3]
    )
